//	TorusGamesGraphics.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include "GeometryGamesMatrix44.h"
#include <math.h>


#define SLICE_PLANE_INSET	0.00001


//	The wall's texture coordinates are used only
//	to algorithmically compute a texture in the fragment function,
//	so in principle they could run over any range we want.
//	In practice it's convenient to let them run from 0 to 1,
//	so the fragment function can use fract() to wrap them.
#define WALL_S_MIN	0.0
#define WALL_S_MAX	1.0
#define WALL_T_MIN	0.0
#define WALL_T_MAX	1.0


const float gMatteColorGammaP3[3] = {0.500, 0.625, 0.750};	//	(1/2, 5/8, 3/4), in gamma-encoded Display P3 coordinates


//	Vertex data for graphical primitives

const TorusGames2DVertexData			gSquareVertices[NUM_SQUARE_VERTICES] =
{
	//	MTKTextureLoader loads images from the Asset Catalog
	//	with bottom-up texture coordinates, in spite of
	//	the Asset Catalog's default "Origin = Top Left".
	//
	//		Proof:  If you let the second texture coordinate
	//		run from 0.0 to 0.5 (instead of from 0.0 to 1.0),
	//		you'll get the bottom half of the image.
	//
	//	Conversely, if you set "Origin = Bottom Left",
	//	and let the second texture coordinate run from 0.0 to 0.5,
	//	you get the top half of the image.
	//
	{{-0.5, -0.5}, {0.0, 0.0}},
	{{-0.5, +0.5}, {0.0, 1.0}},
	{{+0.5, -0.5}, {1.0, 0.0}},
	{{+0.5, +0.5}, {1.0, 1.0}}
};

const TorusGames2DOffsetableVertexData	gLineWithEndcapsOffsettableVertices[NUM_LINE_WITH_ENDCAPS_VERTICES] =
{
	{{-0.5, +0.5}, {0.0, 0.0}, 1.0},
	{{-0.5, -0.5}, {0.0, 1.0}, 1.0},
	{{-0.5, +0.5}, {0.5, 0.0}, 0.0},
	{{-0.5, -0.5}, {0.5, 1.0}, 0.0},
	{{+0.5, +0.5}, {0.5, 0.0}, 0.0},
	{{+0.5, -0.5}, {0.5, 1.0}, 0.0},
	{{+0.5, +0.5}, {1.0, 0.0}, 1.0},
	{{+0.5, -0.5}, {1.0, 1.0}, 1.0},
};

const TorusGames3DWallVertexData		gWallVertices[NUM_WALL_VERTICES] =
{
	{{-0.5, -0.5,  0.0}, {WALL_S_MIN, WALL_T_MIN}, {1.0, 0.0}},
	{{-0.5, +0.5,  0.0}, {WALL_S_MIN, WALL_T_MAX}, {1.0, 0.0}},
	{{+0.5, -0.5,  0.0}, {WALL_S_MAX, WALL_T_MIN}, {1.0, 0.0}},
	{{+0.5, +0.5,  0.0}, {WALL_S_MAX, WALL_T_MAX}, {1.0, 0.0}}
};

const TorusGames3DWallVertexData		gWallWithApertureVertices[NUM_WALL_WITH_APERTURE_VERTICES] =
{
	{{-0.5, -0.5,  0.0}, {WALL_S_MIN, WALL_T_MIN}, {1.0, 0.0}},	//	outer vertex
	{{-0.5, -0.5,  0.0}, {WALL_S_MIN, WALL_T_MIN}, {0.0, 1.0}},	//	inner vertex

	{{+0.5, -0.5,  0.0}, {WALL_S_MAX, WALL_T_MIN}, {1.0, 0.0}},	//	outer vertex
	{{+0.5, -0.5,  0.0}, {WALL_S_MAX, WALL_T_MIN}, {0.0, 1.0}},	//	inner vertex

	{{+0.5, +0.5,  0.0}, {WALL_S_MAX, WALL_T_MAX}, {1.0, 0.0}},	//	outer vertex
	{{+0.5, +0.5,  0.0}, {WALL_S_MAX, WALL_T_MAX}, {0.0, 1.0}},	//	inner vertex

	{{-0.5, +0.5,  0.0}, {WALL_S_MIN, WALL_T_MAX}, {1.0, 0.0}},	//	outer vertex
	{{-0.5, +0.5,  0.0}, {WALL_S_MIN, WALL_T_MAX}, {0.0, 1.0}},	//	inner vertex

	{{-0.5, -0.5,  0.0}, {WALL_S_MIN, WALL_T_MIN}, {1.0, 0.0}},	//	outer vertex
	{{-0.5, -0.5,  0.0}, {WALL_S_MIN, WALL_T_MIN}, {0.0, 1.0}}	//	inner vertex
};

const TorusGames3DPolyhedronVertexData	gCubeVertices[NUM_CUBE_VERTICES] =
{
	{{-1.0, -1.0, -1.0}, {-1.0,  0.0,  0.0}},
	{{-1.0,  1.0, -1.0}, {-1.0,  0.0,  0.0}},
	{{-1.0,  1.0,  1.0}, {-1.0,  0.0,  0.0}},
	{{-1.0, -1.0,  1.0}, {-1.0,  0.0,  0.0}},

	{{ 1.0, -1.0, -1.0}, { 1.0,  0.0,  0.0}},
	{{ 1.0,  1.0, -1.0}, { 1.0,  0.0,  0.0}},
	{{ 1.0,  1.0,  1.0}, { 1.0,  0.0,  0.0}},
	{{ 1.0, -1.0,  1.0}, { 1.0,  0.0,  0.0}},

	{{-1.0, -1.0, -1.0}, { 0.0, -1.0,  0.0}},
	{{-1.0, -1.0,  1.0}, { 0.0, -1.0,  0.0}},
	{{ 1.0, -1.0,  1.0}, { 0.0, -1.0,  0.0}},
	{{ 1.0, -1.0, -1.0}, { 0.0, -1.0,  0.0}},

	{{-1.0,  1.0, -1.0}, { 0.0,  1.0,  0.0}},
	{{-1.0,  1.0,  1.0}, { 0.0,  1.0,  0.0}},
	{{ 1.0,  1.0,  1.0}, { 0.0,  1.0,  0.0}},
	{{ 1.0,  1.0, -1.0}, { 0.0,  1.0,  0.0}},

	{{-1.0, -1.0, -1.0}, { 0.0,  0.0, -1.0}},
	{{ 1.0, -1.0, -1.0}, { 0.0,  0.0, -1.0}},
	{{ 1.0,  1.0, -1.0}, { 0.0,  0.0, -1.0}},
	{{-1.0,  1.0, -1.0}, { 0.0,  0.0, -1.0}},

	{{-1.0, -1.0,  1.0}, { 0.0,  0.0,  1.0}},
	{{ 1.0, -1.0,  1.0}, { 0.0,  0.0,  1.0}},
	{{ 1.0,  1.0,  1.0}, { 0.0,  0.0,  1.0}},
	{{-1.0,  1.0,  1.0}, { 0.0,  0.0,  1.0}},
};
const unsigned int						gCubeNumFacets	= NUM_CUBE_FACETS;
const unsigned short					gCubeFacets[NUM_CUBE_FACETS][3] =
{
	//	Winding order is clockwise when viewed
	//	from outside the cube in a left-handed coordinate system.
	
	{ 1,  0,  2},
	{ 0,  3,  2},

	{ 5,  6,  4},
	{ 6,  7,  4},

	{ 9,  8, 10},
	{ 8, 11, 10},

	{13, 14, 12},
	{14, 15, 12},

	{17, 16, 18},
	{16, 19, 18},

	{21, 22, 20},
	{22, 23, 20},
};

//	The cube skeleton serves as the goal in 3D Maze.
//
//		Note:  These array define the cube skeleton's outer faces only.
//		At run time we make a copy of these arrays and modify it
//		to create the cube skeleton's inner faces.
//
#define OUTER_SIZE	1.0
#define INNER_SIZE	(OUTER_SIZE * GOAL_WINDOW_FRACTION)
const TorusGames3DPolyhedronVertexData	gCubeSkeletonVertices[NUM_CUBE_SKELETON_VERTICES] =
{
	{{-OUTER_SIZE, -OUTER_SIZE, -OUTER_SIZE}, {-1.0,  0.0,  0.0}},
	{{-OUTER_SIZE, -INNER_SIZE, -INNER_SIZE}, {-1.0,  0.0,  0.0}},
	{{-OUTER_SIZE, +OUTER_SIZE, -OUTER_SIZE}, {-1.0,  0.0,  0.0}},
	{{-OUTER_SIZE, +INNER_SIZE, -INNER_SIZE}, {-1.0,  0.0,  0.0}},
	{{-OUTER_SIZE, +OUTER_SIZE, +OUTER_SIZE}, {-1.0,  0.0,  0.0}},
	{{-OUTER_SIZE, +INNER_SIZE, +INNER_SIZE}, {-1.0,  0.0,  0.0}},
	{{-OUTER_SIZE, -OUTER_SIZE, +OUTER_SIZE}, {-1.0,  0.0,  0.0}},
	{{-OUTER_SIZE, -INNER_SIZE, +INNER_SIZE}, {-1.0,  0.0,  0.0}},

	{{+OUTER_SIZE, -INNER_SIZE, -INNER_SIZE}, {+1.0,  0.0,  0.0}},
	{{+OUTER_SIZE, -OUTER_SIZE, -OUTER_SIZE}, {+1.0,  0.0,  0.0}},
	{{+OUTER_SIZE, +INNER_SIZE, -INNER_SIZE}, {+1.0,  0.0,  0.0}},
	{{+OUTER_SIZE, +OUTER_SIZE, -OUTER_SIZE}, {+1.0,  0.0,  0.0}},
	{{+OUTER_SIZE, +INNER_SIZE, +INNER_SIZE}, {+1.0,  0.0,  0.0}},
	{{+OUTER_SIZE, +OUTER_SIZE, +OUTER_SIZE}, {+1.0,  0.0,  0.0}},
	{{+OUTER_SIZE, -INNER_SIZE, +INNER_SIZE}, {+1.0,  0.0,  0.0}},
	{{+OUTER_SIZE, -OUTER_SIZE, +OUTER_SIZE}, {+1.0,  0.0,  0.0}},

	{{-OUTER_SIZE, -OUTER_SIZE, -OUTER_SIZE}, { 0.0, -1.0,  0.0}},
	{{-INNER_SIZE, -OUTER_SIZE, -INNER_SIZE}, { 0.0, -1.0,  0.0}},
	{{-OUTER_SIZE, -OUTER_SIZE, +OUTER_SIZE}, { 0.0, -1.0,  0.0}},
	{{-INNER_SIZE, -OUTER_SIZE, +INNER_SIZE}, { 0.0, -1.0,  0.0}},
	{{+OUTER_SIZE, -OUTER_SIZE, +OUTER_SIZE}, { 0.0, -1.0,  0.0}},
	{{+INNER_SIZE, -OUTER_SIZE, +INNER_SIZE}, { 0.0, -1.0,  0.0}},
	{{+OUTER_SIZE, -OUTER_SIZE, -OUTER_SIZE}, { 0.0, -1.0,  0.0}},
	{{+INNER_SIZE, -OUTER_SIZE, -INNER_SIZE}, { 0.0, -1.0,  0.0}},

	{{-INNER_SIZE, +OUTER_SIZE, -INNER_SIZE}, { 0.0, +1.0,  0.0}},
	{{-OUTER_SIZE, +OUTER_SIZE, -OUTER_SIZE}, { 0.0, +1.0,  0.0}},
	{{-INNER_SIZE, +OUTER_SIZE, +INNER_SIZE}, { 0.0, +1.0,  0.0}},
	{{-OUTER_SIZE, +OUTER_SIZE, +OUTER_SIZE}, { 0.0, +1.0,  0.0}},
	{{+INNER_SIZE, +OUTER_SIZE, +INNER_SIZE}, { 0.0, +1.0,  0.0}},
	{{+OUTER_SIZE, +OUTER_SIZE, +OUTER_SIZE}, { 0.0, +1.0,  0.0}},
	{{+INNER_SIZE, +OUTER_SIZE, -INNER_SIZE}, { 0.0, +1.0,  0.0}},
	{{+OUTER_SIZE, +OUTER_SIZE, -OUTER_SIZE}, { 0.0, +1.0,  0.0}},

	{{-OUTER_SIZE, -OUTER_SIZE, -OUTER_SIZE}, { 0.0,  0.0, -1.0}},
	{{-INNER_SIZE, -INNER_SIZE, -OUTER_SIZE}, { 0.0,  0.0, -1.0}},
	{{+OUTER_SIZE, -OUTER_SIZE, -OUTER_SIZE}, { 0.0,  0.0, -1.0}},
	{{+INNER_SIZE, -INNER_SIZE, -OUTER_SIZE}, { 0.0,  0.0, -1.0}},
	{{+OUTER_SIZE, +OUTER_SIZE, -OUTER_SIZE}, { 0.0,  0.0, -1.0}},
	{{+INNER_SIZE, +INNER_SIZE, -OUTER_SIZE}, { 0.0,  0.0, -1.0}},
	{{-OUTER_SIZE, +OUTER_SIZE, -OUTER_SIZE}, { 0.0,  0.0, -1.0}},
	{{-INNER_SIZE, +INNER_SIZE, -OUTER_SIZE}, { 0.0,  0.0, -1.0}},

	{{-INNER_SIZE, -INNER_SIZE, +OUTER_SIZE}, { 0.0,  0.0, +1.0}},
	{{-OUTER_SIZE, -OUTER_SIZE, +OUTER_SIZE}, { 0.0,  0.0, +1.0}},
	{{+INNER_SIZE, -INNER_SIZE, +OUTER_SIZE}, { 0.0,  0.0, +1.0}},
	{{+OUTER_SIZE, -OUTER_SIZE, +OUTER_SIZE}, { 0.0,  0.0, +1.0}},
	{{+INNER_SIZE, +INNER_SIZE, +OUTER_SIZE}, { 0.0,  0.0, +1.0}},
	{{+OUTER_SIZE, +OUTER_SIZE, +OUTER_SIZE}, { 0.0,  0.0, +1.0}},
	{{-INNER_SIZE, +INNER_SIZE, +OUTER_SIZE}, { 0.0,  0.0, +1.0}},
	{{-OUTER_SIZE, +OUTER_SIZE, +OUTER_SIZE}, { 0.0,  0.0, +1.0}}
};
const unsigned int						gCubeSkeletonNumFacets	= NUM_CUBE_SKELETON_FACETS;
const unsigned short					gCubeSkeletonFacets[NUM_CUBE_SKELETON_FACETS][3] =
{
	//	Winding order is clockwise when viewed
	//	from outside the cube in a left-handed coordinate system.
	
	{ 0,  1,  2},
	{ 1,  3,  2},
	{ 2,  3,  4},
	{ 3,  5,  4},
	{ 4,  5,  6},
	{ 5,  7,  6},
	{ 6,  7,  0},
	{ 7,  1,  0},

	{ 8,  9, 10},
	{ 9, 11, 10},
	{10, 11, 12},
	{11, 13, 12},
	{12, 13, 14},
	{13, 15, 14},
	{14, 15,  8},
	{15,  9,  8},

	{16, 17, 18},
	{17, 19, 18},
	{18, 19, 20},
	{19, 21, 20},
	{20, 21, 22},
	{21, 23, 22},
	{22, 23, 16},
	{23, 17, 16},

	{24, 25, 26},
	{25, 27, 26},
	{26, 27, 28},
	{27, 29, 28},
	{28, 29, 30},
	{29, 31, 30},
	{30, 31, 24},
	{31, 25, 24},

	{32, 33, 34},
	{33, 35, 34},
	{34, 35, 36},
	{35, 37, 36},
	{36, 37, 38},
	{37, 39, 38},
	{38, 39, 32},
	{39, 33, 32},

	{40, 41, 42},
	{41, 43, 42},
	{42, 43, 44},
	{43, 45, 44},
	{44, 45, 46},
	{45, 47, 46},
	{46, 47, 40},
	{47, 41, 40},
	
};

//	The function InitBallMeshes() will populate these arrays.
unsigned int							gBallNumVertices[MAX_BALL_REFINEMENT_LEVEL + 1];
TorusGames3DPolyhedronVertexData		gBallVertices[MAX_BALL_REFINEMENT_LEVEL + 1][MAX_REFINED_BALL_VERTICES];
unsigned int							gBallNumFacets[MAX_BALL_REFINEMENT_LEVEL + 1];
unsigned short							gBallFacets[MAX_BALL_REFINEMENT_LEVEL + 1][MAX_REFINED_BALL_FACETS][3];

//	The function InitTubeMeshes() will populate these arrays.
unsigned int							gTubeNumVertices[MAX_TUBE_REFINEMENT_LEVEL + 1];
TorusGames3DPolyhedronVertexData		gTubeVertices[MAX_TUBE_REFINEMENT_LEVEL + 1][MAX_REFINED_TUBE_VERTICES];
unsigned int							gTubeNumFacets[MAX_TUBE_REFINEMENT_LEVEL + 1];
unsigned short							gTubeFacets[MAX_TUBE_REFINEMENT_LEVEL + 1][MAX_REFINED_TUBE_FACETS][3];

const TorusGames3DPolyhedronVertexData	gSquareSliceVertices[NUM_SQUARE_SLICE_VERTICES] =
{
	//	Square sits at z = 0, with normal vector pointing in negative z direction.
	//
	//	Winding order is clockwise when viewed
	//	from the "outward pointing" side in a left-handed coordinate system.
	//
	{{-1.0, -1.0, 0.0}, {0.0, 0.0, -1.0}},
	{{-1.0, +1.0, 0.0}, {0.0, 0.0, -1.0}},
	{{+1.0, -1.0, 0.0}, {0.0, 0.0, -1.0}},
	{{+1.0, +1.0, 0.0}, {0.0, 0.0, -1.0}}
};

//	The function InitCircularSliceTriangleStrips() will populate these arrays.
unsigned int							gCircularSliceNumVertices[MAX_SLICE_REFINEMENT_LEVEL + 1];
TorusGames3DPolyhedronVertexData		gCircularSliceVertices[MAX_SLICE_REFINEMENT_LEVEL + 1][MAX_NUM_CIRCULAR_SLICE_VERTICES];


//	wall-into-framecell placements
//
//	Each of these six matrices maps a square with vertices
//	at x = ±1/2, y = ±1/2, z = 0 to one of the frame cell's walls.
//
const double	gWallIntoFrameCellPlacements[NUM_FRAME_WALLS][4][4] =
{
	//	left wall
	{
		{ 0.0,  0.0,  1.0,  0.0},
		{ 0.0,  1.0,  0.0,  0.0},
		{-1.0,  0.0,  0.0,  0.0},
		{-0.5,  0.0,  0.0,  1.0}
	},

	//	right wall
	{
		{ 0.0,  0.0, -1.0,  0.0},
		{ 0.0,  1.0,  0.0,  0.0},
		{ 1.0,  0.0,  0.0,  0.0},
		{+0.5,  0.0,  0.0,  1.0}
	},

	//	floor
	{
		{ 1.0,  0.0,  0.0,  0.0},
		{ 0.0,  0.0,  1.0,  0.0},
		{ 0.0, -1.0,  0.0,  0.0},
		{ 0.0, -0.5,  0.0,  1.0}
	},

	//	ceiling
	{
		{-1.0,  0.0,  0.0,  0.0},
		{ 0.0,  0.0,  1.0,  0.0},
		{ 0.0,  1.0,  0.0,  0.0},
		{ 0.0, +0.5,  0.0,  1.0}
	},

	//	near wall
	{
		{-1.0,  0.0,  0.0,  0.0},
		{ 0.0,  1.0,  0.0,  0.0},
		{ 0.0,  0.0, -1.0,  0.0},
		{ 0.0,  0.0, -0.5,  1.0}
	},

	//	far wall
	{
		{ 1.0,  0.0,  0.0,  0.0},
		{ 0.0,  1.0,  0.0,  0.0},
		{ 0.0,  0.0,  1.0,  0.0},
		{ 0.0,  0.0, +0.5,  1.0}
	}
};

//	wall colors
const float	gWallColors[NUM_AVAILABLE_WALL_COLORS][4] =	//	linear sRGB color coordinates
														//	(extended-range is allowed, but not needed)
{
	//	About yellow...
	//		To get a suitably bright yellow,
	//		push the red and green components to 1.25.
	//		If the color were ever shown at full brightness
	//		the 1.25 values would get clamped to 1.0,
	//		but in practice clamping never happens because
	//		the vertex function's tmpDiffuseFactor and/or tmpFogFactor
	//		darken the color substantially at all times.
	//
	
#ifdef GAME_CONTENT_FOR_SCREENSHOT
#warning Disable this code if I ever use a seed other than #51 !
	//	A quick hack to swap some wall colors for use with seed #51,
	//	so that even after rotating the maze a quarter turn,
	//	we still end up with a blue wall at the back and
	//	green walls on the left and right.
	PREMULTIPLY_RGBA(0.500, 0.750, 1.000, 1.000),	//	for left/right walls
	PREMULTIPLY_RGBA(1.250, 1.250, 0.500, 1.000),	//	for floor/ceiling
	PREMULTIPLY_RGBA(0.500, 1.000, 0.750, 1.000),	//	for near/far walls
#else
	PREMULTIPLY_RGBA(0.500, 1.000, 0.750, 1.000),	//	for left/right walls
	PREMULTIPLY_RGBA(1.250, 1.250, 0.500, 1.000),	//	for floor/ceiling
	PREMULTIPLY_RGBA(0.500, 0.750, 1.000, 1.000),	//	for near/far walls
#endif
};


//	gUnusedPlacement is a dummy placement for use with sprites
//	that shouldn't get drawn at all.
//	If such a sprite ever got drawn by accident, it would
//	fill the fundamental domain and be immediately obvious.
//	Set gZeroPlacement for sprites which aren't needed,
//	but will get drawn anyhow.  This hides such sprites
//	from the user without requiring messy communication
//	between the platform-independent C code and
//	the platform-specific user-interface code.
const Placement2D	gUnusedPlacement	= {0.0, 0.0, false, 0.0, 1.0, 1.0},
					gZeroPlacement		= {0.0, 0.0, false, 0.0, 0.0, 0.0};


static void	Get3DPolyhedronPlacementsForSolids(ModelData *md,
				unsigned int aPlacementBufferLength, TorusGames3DPolyhedronPlacementAsCArrays *aPlacementBuffer);
static void	Get3DPolyhedronPlacementsForSlices(ModelData *md,
				unsigned int aRectangularSlicePlacementBufferLength,		TorusGames3DPolyhedronPlacementAsCArrays *aRectangularSlicePlacementBuffer,
				unsigned int aCircularSlicePlacementBufferLength,			TorusGames3DPolyhedronPlacementAsCArrays *aCircularSlicePlacementBuffer,
				unsigned int aClippedEllipticalSlicePlacementBufferLength,	TorusGames3DPolyhedronPlacementAsCArrays *aClippedEllipticalSlicePlacementBuffer);
static void	SetZeroPlacement(TorusGames3DPolyhedronPlacementAsCArrays *anUnusedSlicePlacement);
static void	InitOctahedron(unsigned int aLevel);
static void	SubdivideMesh(unsigned int aSrcLevel);


unsigned int GetNum2DBackgroundTextureRepetitions(
	ModelData	*md)
{
	switch (md->itsGame)
	{
		case GameNone:
			return GetNum2DNoneBackgroundTextureRepetitions();	//	value unused
		
		case Game2DIntro:
			return GetNum2DIntroBackgroundTextureRepetitions();
		
		case Game2DTicTacToe:
			return GetNum2DTicTacToeBackgroundTextureRepetitions();
		
		case Game2DGomoku:
			return GetNum2DGomokuBackgroundTextureRepetitions();
		
		case Game2DMaze:
			return GetNum2DMazeBackgroundTextureRepetitions();
		
		case Game2DCrossword:
			return GetNum2DCrosswordBackgroundTextureRepetitions();
		
		case Game2DWordSearch:
			return GetNum2DWordSearchBackgroundTextureRepetitions();
		
		case Game2DJigsaw:
			return GetNum2DJigsawBackgroundTextureRepetitions();

		case Game2DChess:
			return GetNum2DChessBackgroundTextureRepetitions();
		
		case Game2DPool:
			return GetNum2DPoolBackgroundTextureRepetitions();
		
		case Game2DApples:
			return GetNum2DApplesBackgroundTextureRepetitions();
	
		//	The 3D games don't use a background texture.
		case Game3DTicTacToe:
		case Game3DMaze:
		case NumGameTypes:
			GEOMETRY_GAMES_ABORT("Internal error:  Background texture repetitions requested for invalid game");
			return 1;
	}
}

void Get2DKleinAxisColors(
	ModelData	*md,						//	input
	float		someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	switch (md->itsGame)
	{
		case GameNone:
			Get2DNoneKleinAxisColors(someKleinAxisColors);
			break;

		case Game2DIntro:
			Get2DIntroKleinAxisColors(someKleinAxisColors);
			break;
		
		case Game2DTicTacToe:
			Get2DTicTacToeKleinAxisColors(someKleinAxisColors);
			break;
		
		case Game2DGomoku:
			Get2DGomokuKleinAxisColors(someKleinAxisColors);
			break;
		
		case Game2DMaze:
			Get2DMazeKleinAxisColors(someKleinAxisColors);
			break;
		
		case Game2DCrossword:
			Get2DCrosswordKleinAxisColors(someKleinAxisColors);
			break;
		
		case Game2DWordSearch:
			Get2DWordSearchKleinAxisColors(someKleinAxisColors);
			break;
		
		case Game2DJigsaw:
			Get2DJigsawKleinAxisColors(someKleinAxisColors);
			break;
		
		case Game2DChess:
			Get2DChessKleinAxisColors(someKleinAxisColors);
			break;
		
		case Game2DPool:
			Get2DPoolKleinAxisColors(someKleinAxisColors);
			break;
		
		case Game2DApples:
			Get2DApplesKleinAxisColors(someKleinAxisColors);
			break;
		
		case Game3DTicTacToe:
		case Game3DMaze:
		case NumGameTypes:
		default:
			GEOMETRY_GAMES_ABORT("Internal error:  Klein axis colors requested for invalid game");
	}
}


unsigned int GetNum2DSprites(
	ModelData	*md)
{
	switch (md->itsGame)
	{
		case GameNone:
			return GetNum2DNoneSprites();	//	value unused
		
		case Game2DIntro:
			return GetNum2DIntroSprites();

		case Game2DTicTacToe:
			return GetNum2DTicTacToeSprites();
		
		case Game2DGomoku:
			return GetNum2DGomokuSprites();
		
		case Game2DMaze:
			return GetNum2DMazeSprites();
		
		case Game2DCrossword:
			return GetNum2DCrosswordSprites(md->itsGameOf.Crossword2D.itsPuzzleSize);
		
		case Game2DWordSearch:
			return GetNum2DWordSearchSprites(	md->itsGameOf.WordSearch2D.itsPuzzleSize,
												md->itsGameOf.WordSearch2D.itsNumLinesFound);
		
		case Game2DJigsaw:
			return GetNum2DJigsawSprites(md->itsGameOf.Jigsaw2D.itsSize);

		case Game2DChess:
			return GetNum2DChessSprites();
		
		case Game2DPool:
			return GetNum2DPoolSprites();
		
		case Game2DApples:
			return GetNum2DApplesSprites(md->itsGameOf.Apples2D.itsBoardSize);
		
		//	The 3D games get handled separately.
		case Game3DTicTacToe:
		case Game3DMaze:
		case NumGameTypes:	//	never occurs
			GEOMETRY_GAMES_ABORT("Internal error:  Number of 2D sprites requested for invalid game");
			return 0;
	}
}

void Get2DSpritePlacements(
	ModelData		*md,				//	input
	unsigned int	aNumSprites,		//	input
	Placement2D		*aPlacementBuffer)	//	output;  big enough to hold aNumSprites Placement2D's
{
	switch (md->itsGame)
	{
		case GameNone:
			Get2DNoneSpritePlacements(md, aNumSprites, aPlacementBuffer);
			break;
		
		case Game2DIntro:
			Get2DIntroSpritePlacements(md, aNumSprites, aPlacementBuffer);
			break;

		case Game2DTicTacToe:
			Get2DTicTacToeSpritePlacements(md, aNumSprites, aPlacementBuffer);
			break;
		
		case Game2DGomoku:
			Get2DGomokuSpritePlacements(md, aNumSprites, aPlacementBuffer);
			break;
		
		case Game2DMaze:
			Get2DMazeSpritePlacements(md, aNumSprites, aPlacementBuffer);
			break;
		
		case Game2DCrossword:
			Get2DCrosswordSpritePlacements(md, aNumSprites, aPlacementBuffer);
			break;
		
		case Game2DWordSearch:
			Get2DWordSearchSpritePlacements(md, aNumSprites, aPlacementBuffer);
			break;
		
		case Game2DJigsaw:
			Get2DJigsawSpritePlacementsOrTexturePlacements(md, aNumSprites, aPlacementBuffer, NULL);
			break;

		case Game2DChess:
			Get2DChessSpritePlacements(md, aNumSprites, aPlacementBuffer);
			break;
		
		case Game2DPool:
			Get2DPoolSpritePlacements(md, aNumSprites, aPlacementBuffer);
			break;
		
		case Game2DApples:
			Get2DApplesSpritePlacements(md, aNumSprites, aPlacementBuffer);
			break;
		
		//	The 3D games get handled separately.
		case Game3DTicTacToe:
		case Game3DMaze:
		case NumGameTypes:	//	never occurs
			GEOMETRY_GAMES_ABORT("Internal error:  Sprite placements requested for invalid game");
			break;
	}
}


void Make3DFogDistances(
	ModelData	*md,
//	__fp16		*aFogBegin,	//	as __fp16 for GPU
	__fp16		*aFogEnd,	//	as __fp16 for GPU
	__fp16		*aFogScale)	//	as __fp16 for GPU
{
	double	theFogBegin,
			theFogEnd,
			theFogRange,
			theFogScale;

	//	The fog could start at radius 0.5, which corresponds to the center
	//	of the frame cell's nearest face when the frame cell is in its default position,
	//	and run out to the CLIPPING_RADIUS_FOR_REPEATING_VIEW.
	theFogBegin	= 0.5;
	theFogEnd	= CLIPPING_RADIUS_FOR_REPEATING_VIEW;
	theFogRange	= theFogEnd - theFogBegin;

	//	However, that simple fogging makes objects close to the clipping sphere
	//	so dark that the user can't see them at all.  So let's extend the fog range
	//	slightly beyond the clipping sphere.  Adding an extra 1/8 to the range
	//	works well, in the sense that the fog still effectively hides objects
	//	that are passing through the clipping sphere, while nevertheless showing
	//	objects that sit just a little closer than that.
	theFogRange	*= 1.125;
	theFogEnd	= theFogBegin + theFogRange;

	//	The one exception is that during Simulation3DResetGameAsTiling
	//	we shift the fog, to darken the scene during the game reset.
	if (md->itsSimulationStatus == Simulation3DResetGameAsTilingPart1
	 || md->itsSimulationStatus == Simulation3DResetGameAsTilingPart2
	 || md->itsSimulationStatus == Simulation3DResetGameAsTilingPart3)
	{
//		theFogBegin	-= md->its3DResetGameTilingFogShift * theFogRange;
		theFogEnd	-= md->its3DResetGameTilingFogShift * theFogRange;
	}
	
	//	Pre-compute the scale here, to avoid having to do
	//	that same division over and over in the GPU function.
	theFogScale = 1.0 / theFogRange;
	
	//	Return the results.
//	*aFogBegin	= (__fp16) theFogBegin;
	*aFogEnd	= (__fp16) theFogEnd;
	*aFogScale	= (__fp16) theFogScale;
}

double *Make3DTexCoordShift(						//	returns aTexCoordShift as a convenience
	double			a3DTilingIntoFrameCell[4][4],	//	input
	unsigned int	aWallIndex,						//	input;  ∈ {0,1,2,3,4,5}
	double			aTexCoordShift[2])				//	output;  (Δs, Δt)
{
	unsigned int	i,
					j;
	double			*theTilingShiftInFrameCellCoordinates,	//	alias for a 3-element vector
					theTilingShiftInWallCoordinates[3];
	
	GEOMETRY_GAMES_ASSERT(
		aWallIndex < NUM_FRAME_WALLS,
		"invalid wall index");
	
	//	The last row of a3DTilingIntoFrameCell tells where
	//	the tiling's origin sits in frame cell coordinates.
	//	By default the tiling's origin sits at the frame cell's
	//	origin (0,0,0,1), but as the user scrolls the game,
	//	the former will move away from the latter.
	//	Let theTilingShiftInFrameCellCoordinates record the difference vector.
	//	The first three elements of a3DTilingIntoFrameCell's last row
	//	(ignoring the fourth element in that row!) already contain
	//	exactly the difference vector we need.
	//
	//		Note:  We use the last row here because we're
	//		working with a left-to-right matrix convention.
	//		If we instead used a right-to-left matrix convention,
	//		then we'd need the last column instead of the last row.
	//
	theTilingShiftInFrameCellCoordinates = a3DTilingIntoFrameCell[3];
	
	//	gWallIntoFrameCellPlacements[aWallIndex] maps wall coordinates to frame cell coordinates.
	//	Here we want to go the other way, so we'll need to use its inverse.
	//	Fortunately, for a rigid motion like this (an element of SO(3))
	//	the inverse is simply the transpose.
	//
	for (i = 0; i < 3; i++)	//	Compute all three elements just for good form,
							//		even though we'll need only the first two.
	{
		theTilingShiftInWallCoordinates[i] = 0.0;
		
		for (j = 0; j < 3; j++)
		{
			theTilingShiftInWallCoordinates[i] += theTilingShiftInFrameCellCoordinates[j]
												* gWallIntoFrameCellPlacements[aWallIndex][i][j];
																				//	note transpose!
		}
	}
	
	//	As the tiling's origin moves rightward in frame cell coordinates (x increases)
	//	the wall's texture coordinate s must decrease.
	//	As the tiling's origin moves upward in frame cell coordinates (y increases)
	//	the wall's texture coordinate t must decrease.
	aTexCoordShift[0] = - theTilingShiftInWallCoordinates[0];
	aTexCoordShift[1] = - theTilingShiftInWallCoordinates[1];
	
	//	Note:  theTilingShiftInWallCoordinates[2] isn't needed.
	
	return aTexCoordShift;
}


unsigned int GetNum3DPolyhedra(
	ModelData	*md)
{
	//	The buffer that Get3DPolyhedronPlacements() will fill consists of four parts.
	//	A separate function defines the size of each part,
	//	so that the rendering code can access them independently,
	//	and most importantly so that if we ever change the sizes
	//	of the four parts, the rendering code will remain correct.
	
	return	GetNum3DPolyhedra_BufferPart1_Solids(md)
		  + GetNum3DPolyhedra_BufferPart2_RectangularSlices(md)
		  + GetNum3DPolyhedra_BufferPart3_CircularSlices(md)
		  + GetNum3DPolyhedra_BufferPart4_ClippedEllipticalSlices(md);
}

unsigned int GetNum3DPolyhedra_BufferPart1_Solids(
	ModelData	*md)
{
	//	Get3DPolyhedronPlacements() returns the marker and win line placements
	//	in a well-defined order in the first part of the buffer.

	switch (md->itsGame)
	{
		case Game3DTicTacToe:
			return GetNum3DTicTacToePolyhedra_BufferPart1_Solids(md);

		case Game3DMaze:
			return GetNum3DMazePolyhedra_BufferPart1_Solids(md);
		
		//	The 2D games get handled separately.
		case GameNone:
		case Game2DIntro:
		case Game2DTicTacToe:
		case Game2DGomoku:
		case Game2DMaze:
		case Game2DCrossword:
		case Game2DWordSearch:
		case Game2DJigsaw:
		case Game2DChess:
		case Game2DPool:
		case Game2DApples:
		case NumGameTypes:	//	never occurs
			GEOMETRY_GAMES_ABORT("Internal error:  Number of part 1 polyhedra requested for invalid game");
			return 0;
	}
}

unsigned int GetNum3DPolyhedra_BufferPart2_RectangularSlices(
	ModelData	*md)
{
	//	The second part of the buffer is reserved for the "rectangular slices",
	//	for example the cross section of a cubical Tic-Tac-Toe marker
	//	or a Maze tube that runs along a frame cell wall.
	//	Get3DPolyhedronPlacements() treats this second part of the buffer
	//	as a single homogeneous block, into which it writes as many
	//	slice placements as necessary, in no particular order.
	//	It fills the unused space in this part of the buffer with all-zero matrices,
	//	so that the rendering code will know which entries in this part
	//	of the buffer are in use and which are not.  More precisely:
	//
	//		if a placement matrix M   is   in use, then M[3][3] is guaranteed to be 1.0
	//		if a placement matrix M is not in use, then M[3][3] is guaranteed to be 0.0
	//

	switch (md->itsGame)
	{
		case Game3DTicTacToe:
			return GetNum3DTicTacToePolyhedra_BufferPart2_RectangularSlices(md);

		case Game3DMaze:
			return GetNum3DMazePolyhedra_BufferPart2_RectangularSlices(md);
		
		//	The 2D games get handled separately.
		case GameNone:
		case Game2DIntro:
		case Game2DTicTacToe:
		case Game2DGomoku:
		case Game2DMaze:
		case Game2DCrossword:
		case Game2DWordSearch:
		case Game2DJigsaw:
		case Game2DChess:
		case Game2DPool:
		case Game2DApples:
		case NumGameTypes:	//	never occurs
			GEOMETRY_GAMES_ABORT("Internal error:  Number of part 2 polyhedra requested for invalid game");
			return 0;
	}
}

unsigned int GetNum3DPolyhedra_BufferPart3_CircularSlices(
	ModelData	*md)
{
	//	The third part of the buffer is reserved for the "circular slices",
	//	for example the cross section of a spherical Tic-Tac-Toe marker
	//	or a Maze tube that runs perpendicular to a frame cell wall.
	//	This part of the buffer is treated as a single homogeneous block,
	//	as explained in GetNum3DPolyhedra_BufferPart2_RectangularSlices() above.

	switch (md->itsGame)
	{
		case Game3DTicTacToe:
			return GetNum3DTicTacToePolyhedra_BufferPart3_CircularSlices(md);

		case Game3DMaze:
			return GetNum3DMazePolyhedra_BufferPart3_CircularSlices(md);
		
		//	The 2D games get handled separately.
		case GameNone:
		case Game2DIntro:
		case Game2DTicTacToe:
		case Game2DGomoku:
		case Game2DMaze:
		case Game2DCrossword:
		case Game2DWordSearch:
		case Game2DJigsaw:
		case Game2DChess:
		case Game2DPool:
		case Game2DApples:
		case NumGameTypes:	//	never occurs
			GEOMETRY_GAMES_ABORT("Internal error:  Number of part 3 polyhedra requested for invalid game");
			return 0;
	}
}

unsigned int GetNum3DPolyhedra_BufferPart4_ClippedEllipticalSlices(
	ModelData	*md)
{
	//	The fourth part of the buffer is reserved for the (non-circular) "elliptical slices".
	//	It's currently used only for a Tic-Tac-Toe win-line tube's (non-circular) elliptical cross sections,
	//	which may sometimes need additional clipping at either end of the tube.
	//	(But all non-circular elliptical slices go here, whether they actually need the clipping or not.)
	//	This part of the buffer is treated as a single homogeneous block,
	//	as explained in GetNum3DPolyhedra_BufferPart2_RectangularSlices() above.

	switch (md->itsGame)
	{
		case Game3DTicTacToe:
			return GetNum3DTicTacToePolyhedra_BufferPart4_ClippedEllipticalSlices(md);

		case Game3DMaze:
			return GetNum3DMazePolyhedra_BufferPart4_ClippedEllipticalSlices(md);
		
		//	The 2D games get handled separately.
		case GameNone:
		case Game2DIntro:
		case Game2DTicTacToe:
		case Game2DGomoku:
		case Game2DMaze:
		case Game2DCrossword:
		case Game2DWordSearch:
		case Game2DJigsaw:
		case Game2DChess:
		case Game2DPool:
		case Game2DApples:
		case NumGameTypes:	//	never occurs
			GEOMETRY_GAMES_ABORT("Internal error:  Number of part 3 polyhedra requested for invalid game");
			return 0;
	}
}


void Get3DPolyhedronPlacements(
	ModelData									*md,					//	input
	unsigned int								aPlacementBufferLength,	//	input (buffer length in placements, not bytes)
	TorusGames3DPolyhedronPlacementAsCArrays	*aPlacementBuffer)		//	output buffer with space for aPlacementBufferLength placements
{
	unsigned int								theSolidBufferLength,					//	in placements, not bytes
												theRectangularSliceBufferLength,		//	in placements, not bytes
												theCircularSliceBufferLength,			//	in placements, not bytes
												theClippedEllipticalSliceBufferLength;	//	in placements, not bytes
	TorusGames3DPolyhedronPlacementAsCArrays	*theSolidBuffer,
												*theRectangularSliceBuffer,
												*theCircularSliceBuffer,
												*theClippedEllipticalSliceBuffer;

	GEOMETRY_GAMES_ASSERT(
		aPlacementBufferLength == GetNum3DPolyhedra(md),
		"Internal error:  wrong buffer size");

	theSolidBufferLength					= GetNum3DPolyhedra_BufferPart1_Solids(md);
	theRectangularSliceBufferLength			= GetNum3DPolyhedra_BufferPart2_RectangularSlices(md);
	theCircularSliceBufferLength			= GetNum3DPolyhedra_BufferPart3_CircularSlices(md);
	theClippedEllipticalSliceBufferLength	= GetNum3DPolyhedra_BufferPart4_ClippedEllipticalSlices(md);

	GEOMETRY_GAMES_ASSERT(
		aPlacementBufferLength == theSolidBufferLength
								+ theRectangularSliceBufferLength
								+ theCircularSliceBufferLength
								+ theClippedEllipticalSliceBufferLength,
		"Internal error:  inconsistent buffer part sizes");
	
	theSolidBuffer					= aPlacementBuffer;
	theRectangularSliceBuffer		= theSolidBuffer            + theSolidBufferLength;
	theCircularSliceBuffer			= theRectangularSliceBuffer + theRectangularSliceBufferLength;
	theClippedEllipticalSliceBuffer	= theCircularSliceBuffer    + theCircularSliceBufferLength;

	//	Solid polyhedra
	Get3DPolyhedronPlacementsForSolids(	md,
										theSolidBufferLength, theSolidBuffer);

	//	Cross-sectional slices, where the solid polyhedra get clipped
	//	to the frame cell (ViewBasicLarge) or the near clip plane (ViewRepeating).
	Get3DPolyhedronPlacementsForSlices(	md,
										theRectangularSliceBufferLength,		theRectangularSliceBuffer,
										theCircularSliceBufferLength,			theCircularSliceBuffer,
										theClippedEllipticalSliceBufferLength,	theClippedEllipticalSliceBuffer);
}

static void Get3DPolyhedronPlacementsForSolids(
	ModelData									*md,					//	input
	unsigned int								aPlacementBufferLength,	//	input (buffer length in placements, not bytes)
	TorusGames3DPolyhedronPlacementAsCArrays	*aPlacementBuffer)		//	output buffer with space for aPlacementBufferLength placements
{
	switch (md->itsGame)
	{
		case Game3DTicTacToe:
			Get3DTicTacToePolyhedronPlacementsForSolids(md, aPlacementBufferLength, aPlacementBuffer);
			break;

		case Game3DMaze:
			Get3DMazePolyhedronPlacementsForSolids(md, aPlacementBufferLength, aPlacementBuffer);
			break;
		
		//	The 2D games get handled separately.
		case GameNone:
		case Game2DIntro:
		case Game2DTicTacToe:
		case Game2DGomoku:
		case Game2DMaze:
		case Game2DCrossword:
		case Game2DWordSearch:
		case Game2DJigsaw:
		case Game2DChess:
		case Game2DPool:
		case Game2DApples:
		case NumGameTypes:	//	never occurs
			GEOMETRY_GAMES_ABORT("Internal error:  Polyhedron placements requested for invalid game");
			break;
	}
}

static void Get3DPolyhedronPlacementsForSlices(
	ModelData									*md,											//	input
	unsigned int								aRectangularSlicePlacementBufferLength,			//	input (buffer length in placements, not bytes)
	TorusGames3DPolyhedronPlacementAsCArrays	*aRectangularSlicePlacementBuffer,				//	output buffer with space for aRectangularSlicePlacementBufferLength placements
	unsigned int								aCircularSlicePlacementBufferLength,			//	input (buffer length in placements, not bytes)
	TorusGames3DPolyhedronPlacementAsCArrays	*aCircularSlicePlacementBuffer,					//	output buffer with space for aCircularSlicePlacementBufferLength placements
	unsigned int								aClippedEllipticalSlicePlacementBufferLength,	//	input (buffer length in placements, not bytes)
	TorusGames3DPolyhedronPlacementAsCArrays	*aClippedEllipticalSlicePlacementBuffer)		//	output buffer with space for aClippedEllipticalSlicePlacementBufferLength placements
{
	double										theFrameCellIntoWorld[4][4];
	unsigned int								i;
	const double								*theWallNormalsInFrameCell[NUM_FRAME_WALLS];
	bool										theWallIsTransparent[NUM_FRAME_WALLS];	//	Does the user see this wall from the outside of the frame cell?
	double										theFrameCellIntoTiling[4][4],
												theFrameCellCenterInTiling[4],
												theWallNormalsInTiling[NUM_FRAME_WALLS][4];
	unsigned int								theNumUnusedRectangularSlicePlacements;
	TorusGames3DPolyhedronPlacementAsCArrays	*theUnusedRectangularSlicePlacements;
	unsigned int								theNumUnusedCircularSlicePlacements;
	TorusGames3DPolyhedronPlacementAsCArrays	*theUnusedCircularSlicePlacements;
	unsigned int								theNumUnusedClippedEllipticalSlicePlacements;
	TorusGames3DPolyhedronPlacementAsCArrays	*theUnusedClippedEllipticalSlicePlacements;
	signed int									x,
												y,
												z;
	double										theGameCellIntoTiling[4][4],
												theGameCellIntoFrameCell[4][4],
												theTilingIntoGameCell[4][4],
												theFrameCellCenterInGameCell[4],
												theWallNormalInGameCell[4];
	unsigned int								theAxis;						//	= 0, 1 or 2 for x, y or z
	bool										thePositiveAxisFlag = false;	//	initialize to suppress compiler warning
	double										theIntercept,
												theRotationalPartOfSlicePlacement[4][4];

	static /*const*/ double	theFrameCellCenterInFrameCell[4] = {0.0, 0.0, 0.0, 1.0};
	
	//	This game-independent function does all the setup
	//	for finding cross sectional slices and then,
	//	in an innermost loop, calls a game-specific function
	//	to compute the actual cross sections.

	//	Draw the slices at distance 0.49999 to guarantee
	//	that they don't get clipped away due to numerical error.
	const double	theSlicePlaneDistance	= 0.5 - SLICE_PLANE_INSET;

	//	Convert its3DFrameCellIntoWorld from a spin vector in Spin(3)
	//	to a 4×4 matrix in SO(3), padded out with zero translational part.
	RealizeIsometryAs4x4MatrixInSO3(&md->its3DFrameCellIntoWorld, theFrameCellIntoWorld);

	//	Each wall placement's third row is exactly
	//	that wall's outward-pointing normal vector.
	for (i = 0; i < NUM_FRAME_WALLS; i++)
		theWallNormalsInFrameCell[i] = gWallIntoFrameCellPlacements[i][2];

	//	Which of the frame cell's six faces does the user see from the outside?
	for (i = 0; i < NUM_FRAME_WALLS; i++)
	{
		theWallIsTransparent[i] = Wall3DShouldBeTransparent(
								theWallNormalsInFrameCell[i],
								theFrameCellIntoWorld,
								(	md->itsSimulationStatus == Simulation3DResetGameAsDomainPart1
								 ||	md->itsSimulationStatus == Simulation3DResetGameAsDomainPart2
								 ||	md->itsSimulationStatus == Simulation3DResetGameAsDomainPart3) ?
									md->its3DResetGameScaleFactor : 1.0);
	}

	//	While the reset game animation is running, some walls
	//	that would otherwise be transparent will be drawn opaque.
	//	Don't draw clip plane slices on them.
	if (md->itsSimulationStatus == Simulation3DResetGameAsDomainPart1
	 ||	md->itsSimulationStatus == Simulation3DResetGameAsDomainPart2
	 ||	md->itsSimulationStatus == Simulation3DResetGameAsDomainPart3)
	{
		for (i = 0; i < NUM_FRAME_WALLS; i++)
			if (md->its3DResetGameOpaqueWalls[i])
				theWallIsTransparent[i] = false;
	}
	
	//	Convert the frame cell's center and wall normals
	//	from frame cell coordinates to tiling coordinates.
	Matrix44GeometricInverse(md->its3DTilingIntoFrameCell, theFrameCellIntoTiling);
	Matrix44RowVectorTimesMatrix(	theFrameCellCenterInFrameCell,
									theFrameCellIntoTiling,
									theFrameCellCenterInTiling);
	for (i = 0; i < NUM_FRAME_WALLS; i++)
	{
		Matrix44RowVectorTimesMatrix(	theWallNormalsInFrameCell[i],
										theFrameCellIntoTiling,
										theWallNormalsInTiling[i]);
	}
	
	//	Keep track of the remaining space in each buffer.
	//	As the game-specific code fills these buffers,
	//	it will increment the buffer pointers and decrement the counts.

	theNumUnusedRectangularSlicePlacements			= aRectangularSlicePlacementBufferLength;
	theUnusedRectangularSlicePlacements				= aRectangularSlicePlacementBuffer;

	theNumUnusedCircularSlicePlacements				= aCircularSlicePlacementBufferLength;
	theUnusedCircularSlicePlacements				= aCircularSlicePlacementBuffer;

	theNumUnusedClippedEllipticalSlicePlacements	= aClippedEllipticalSlicePlacementBufferLength;
	theUnusedClippedEllipticalSlicePlacements		= aClippedEllipticalSlicePlacementBuffer;

	//	Iterate over a 3×3×3 tiling of the nearest copies of the game cell,
	//	examining the images of each object and saving a placement matrix
	//	for the cross section of each object that intersects a visible frame cell wall.
	for (x = -1; x <= +1; x++)
	{
		for (y = -1; y <= +1; y++)
		{
			for (z = -1; z <= +1; z++)
			{
				Make3DGameCellIntoTiling(theGameCellIntoTiling, x, y, z, md->itsTopology);
				
				Matrix44Product(theGameCellIntoTiling, md->its3DTilingIntoFrameCell, theGameCellIntoFrameCell);

				Matrix44GeometricInverse(theGameCellIntoTiling, theTilingIntoGameCell);
				Matrix44RowVectorTimesMatrix(	theFrameCellCenterInTiling,
												theTilingIntoGameCell,
												theFrameCellCenterInGameCell);
				
				for (i = 0; i < NUM_FRAME_WALLS; i++)
				{
					if (theWallIsTransparent[i])
					{
						Matrix44RowVectorTimesMatrix(	theWallNormalsInTiling[i],
														theTilingIntoGameCell,
														theWallNormalInGameCell);

						//	Even after all coordinate transformations, theWallNormalInGameCell
						//	should be one of the six vectors (±1,0,0), (0,±1,0), (0,0,±1).
						//	Check which axis it points along, and in which direction.
						for (theAxis = 0; theAxis < 3; theAxis++)
						{
							if (theWallNormalInGameCell[theAxis] == -1.0)
							{
								thePositiveAxisFlag = false;
								break;
							}
							if (theWallNormalInGameCell[theAxis] == +1.0)
							{
								thePositiveAxisFlag = true;
								break;
							}
						}
						GEOMETRY_GAMES_ASSERT(theAxis < 3, "theWallNormalsInGameCell[i] is invalid.");
						
						//	What is the slice plane's coordinate along theAxis?
						theIntercept = theFrameCellCenterInGameCell[theAxis]
									 + (thePositiveAxisFlag ? +theSlicePlaneDistance : -theSlicePlaneDistance);

						//	Compute the rotational part of the potential slice placements in the game cell.
						Matrix44Identity(theRotationalPartOfSlicePlacement);	//	The slice faces the -z axis by default.
						if (thePositiveAxisFlag)								//	If viewing slice from positive axis,
							HalfTurnY(theRotationalPartOfSlicePlacement);		//		rotate it to see front face.
						if (theAxis == 0)										//	Rotate to face correct direction.
							RotateXYZ(theRotationalPartOfSlicePlacement);
						if (theAxis == 1)
							RotateZYX(theRotationalPartOfSlicePlacement);

						//	Let the game-specific code compute the placement matrices for the slices
						//	of any objects that intersect the plane with the given axis and intercept.
						//
						//		Note:  If we pass theFrameCellCenterInGameCell to the game-specific code,
						//		it may cull away objects that lie wholly outside the frame cell,
						//		for improved performance.  (If it didn't cull them, they'd get clipped away
						//		later on, but only after the GPU had processed all their vertices.)
						//
						switch (md->itsGame)
						{
							case Game3DTicTacToe:
								Get3DTicTacToePolyhedronPlacementsForOneSlice(
									md, theAxis, theIntercept,
									theFrameCellCenterInGameCell, theRotationalPartOfSlicePlacement, theGameCellIntoFrameCell,
									&theNumUnusedRectangularSlicePlacements,		&theUnusedRectangularSlicePlacements,
									&theNumUnusedCircularSlicePlacements,			&theUnusedCircularSlicePlacements,
									&theNumUnusedClippedEllipticalSlicePlacements,	&theUnusedClippedEllipticalSlicePlacements);
								break;

							case Game3DMaze:
								Get3DMazePolyhedronPlacementsForOneSlice(
									md, theAxis, theIntercept,
									theFrameCellCenterInGameCell, theRotationalPartOfSlicePlacement, theGameCellIntoFrameCell,
									&theNumUnusedRectangularSlicePlacements,		&theUnusedRectangularSlicePlacements,
									&theNumUnusedCircularSlicePlacements,			&theUnusedCircularSlicePlacements);
								break;
							
							//	The 2D games get handled separately.
							case GameNone:
							case Game2DIntro:
							case Game2DTicTacToe:
							case Game2DGomoku:
							case Game2DMaze:
							case Game2DCrossword:
							case Game2DWordSearch:
							case Game2DJigsaw:
							case Game2DChess:
							case Game2DPool:
							case Game2DApples:
							case NumGameTypes:	//	never occurs
								GEOMETRY_GAMES_ABORT("Internal error:  Polyhedron placements requested for invalid game");
								break;
						}
					}
				}
			}
		}
	}
	
	//	The game-specific functions should have already taken responsibility
	//	for checking that buffer space is available before writing each new placement matrix,
	//	but just be doubly safe...
	GEOMETRY_GAMES_ASSERT(
		theUnusedRectangularSlicePlacements			<= aRectangularSlicePlacementBuffer			+ aRectangularSlicePlacementBufferLength,
		"Internal error:  buffer overrun in aRectangularSlicePlacementBuffer");
	GEOMETRY_GAMES_ASSERT(
		theUnusedCircularSlicePlacements			<= aCircularSlicePlacementBuffer			+ aCircularSlicePlacementBufferLength,
		"Internal error:  buffer overrun in aCircularSlicePlacementBuffer"   );
	GEOMETRY_GAMES_ASSERT(
		theUnusedClippedEllipticalSlicePlacements	<= aClippedEllipticalSlicePlacementBuffer	+ aClippedEllipticalSlicePlacementBufferLength,
		"Internal error:  buffer overrun in aClippedEllipticalSlicePlacementBuffer"   );

	//	Also a quick consistency check (keep in mind that the counts are unsigned integers, which wrap if they "go negative").
	GEOMETRY_GAMES_ASSERT(
		theUnusedRectangularSlicePlacements			+ theNumUnusedRectangularSlicePlacements		== aRectangularSlicePlacementBufferLength		+ aRectangularSlicePlacementBuffer,
		"Internal error:  inconsistency in aRectangularSlicePlacementBuffer");
	GEOMETRY_GAMES_ASSERT(
		theUnusedCircularSlicePlacements			+ theNumUnusedCircularSlicePlacements			== aCircularSlicePlacementBufferLength			+ aCircularSlicePlacementBuffer,
		"Internal error:  inconsistency in aCircularSlicePlacementBuffer"   );
	GEOMETRY_GAMES_ASSERT(
		theUnusedClippedEllipticalSlicePlacements	+ theNumUnusedClippedEllipticalSlicePlacements	== aClippedEllipticalSlicePlacementBufferLength	+ aClippedEllipticalSlicePlacementBuffer,
		"Internal error:  inconsistency in aClippedEllipticalSlicePlacementBuffer"   );
	
	//	Fill any unused placements with zero-matrices,
	//	so that the rendering code will know which placement matrices
	//	are in use and which aren't.
	while (theNumUnusedRectangularSlicePlacements > 0)
	{
		SetZeroPlacement(theUnusedRectangularSlicePlacements);
		theUnusedRectangularSlicePlacements++;
		theNumUnusedRectangularSlicePlacements--;
	}
	while (theNumUnusedCircularSlicePlacements > 0)
	{
		SetZeroPlacement(theUnusedCircularSlicePlacements);
		theUnusedCircularSlicePlacements++;
		theNumUnusedCircularSlicePlacements--;
	}
	while (theNumUnusedClippedEllipticalSlicePlacements > 0)
	{
		SetZeroPlacement(theUnusedClippedEllipticalSlicePlacements);
		theUnusedClippedEllipticalSlicePlacements++;
		theNumUnusedClippedEllipticalSlicePlacements--;
	}
}

static void SetZeroPlacement(
	TorusGames3DPolyhedronPlacementAsCArrays	*anUnusedSlicePlacement)
{
	unsigned int	i;
	
	for (i = 0; i < 3; i++)
		anUnusedSlicePlacement->itsDilation[i] = 0.0;

	Matrix44Zero(anUnusedSlicePlacement->itsIsometricPlacement);

	for (i = 0; i < 4; i++)
		anUnusedSlicePlacement->itsExtraClippingCovector[i] = 0.0;
}


void Get3DAxisAlignedBoundingBox(
	ModelData	*md,									//	input
	double		someBoundingBoxCornersInGameCell[2][4])	//	output;  any pair of diametrically opposite corners
{
	//	Let the game specify an axis-aligned bounding box
	//	that encloses the game's entire content, including
	//	any content that might extend beyond the game cell proper.
	//	For example, in the Maze some of the struts will cross the game cell walls,
	//	and the slider may occasionally straddle a game cell wall.
	//
	//	To know an axis-aligned bounding box, it suffices
	//	to know a pair of diametrically opposite corners.
	//
	switch (md->itsGame)
	{
		case Game3DTicTacToe:
			Get3DTicTacToeAxisAlignedBoundingBox(md, someBoundingBoxCornersInGameCell);
			break;

		case Game3DMaze:
			Get3DMazeAxisAlignedBoundingBox(md, someBoundingBoxCornersInGameCell);
			break;
		
		//	The 2D games don't use bounding boxes.
		case GameNone:
		case Game2DIntro:
		case Game2DTicTacToe:
		case Game2DGomoku:
		case Game2DMaze:
		case Game2DCrossword:
		case Game2DWordSearch:
		case Game2DJigsaw:
		case Game2DChess:
		case Game2DPool:
		case Game2DApples:
		case NumGameTypes:	//	never occurs
			GEOMETRY_GAMES_ABORT("Internal error:  Bounding box requested for invalid game");
			break;
	}
}


void InitBallMeshes(void)
{
	unsigned int	i;

	static bool		theBallMeshesHaveBeenInitialized	= false;
	
	//	The meshes need be initialized only once.
	if (theBallMeshesHaveBeenInitialized)
		return;
	
	//	Construct an octahedron for the base level.
	InitOctahedron(0);
	
	//	Subdivide each mesh to get the next one in the series.
	for (i = 0; i <= MAX_BALL_REFINEMENT_LEVEL - 1; i++)
		SubdivideMesh(i);
	
	//	The one-time mesh initialization is now complete.
	//	No need to repeat it if this function gets called again.
	theBallMeshesHaveBeenInitialized = true;
}

static void InitOctahedron(
	unsigned int	aLevel)	//	in practice, aLevel = 0
{
	unsigned int	i,
					j;

	//	Triangulate the octahedron as
	//
	//		1-----2-----1
	//		| \ / | \ / |
	//		|  4  |  5  |
	//		| / \ | / \ |
	//		3-----0-----3
	//
	static const TorusGames3DPolyhedronVertexData	v[6] =
													{
														{{-1.0,  0.0,  0.0 }, {-1.0,  0.0,  0.0 }},
														{{ 1.0,  0.0,  0.0 }, { 1.0,  0.0,  0.0 }},
														{{ 0.0, -1.0,  0.0 }, { 0.0, -1.0,  0.0 }},
														{{ 0.0,  1.0,  0.0 }, { 0.0,  1.0,  0.0 }},
														{{ 0.0,  0.0, -1.0 }, { 0.0,  0.0, -1.0 }},
														{{ 0.0,  0.0,  1.0 }, { 0.0,  0.0,  1.0 }}
													};
	static const unsigned short						f[8][3] =
													{
														//	Winding order is clockwise when viewed
														//	from outside the cube in a left-handed coordinate system.
														
														//	For the 1/8 sphere at x ≥ 0, y ≥ 0, z ≥ 0,
														//	use only this face (and its subdivisions):
														{5, 1, 3},

														//	For the 1/4 sphere at x ≥ 0, y ≥ 0,
														//	use the previous face along with this one
														//	(and their subdivisions):
														{4, 3, 1},

														//	For the 1/2 sphere at x ≥ 0,
														//	use the previous faces along with these ones
														//	(and their subdivisions):
														{4, 1, 2},
														{5, 2, 1},
														
														//	For the full sphere,
														//	use the previous faces along with these ones
														//	(and their subdivisions):
														{4, 0, 3},
														{4, 2, 0},
														{5, 3, 0},
														{5, 0, 2}
													};

	GEOMETRY_GAMES_ASSERT(aLevel <= MAX_BALL_REFINEMENT_LEVEL, "invalid level");
	GEOMETRY_GAMES_ASSERT(MAX_REFINED_BALL_VERTICES >= 6, "array too small to hold octahedron vertices");
	GEOMETRY_GAMES_ASSERT(MAX_REFINED_BALL_FACETS   >= 8, "array too small to hold octahedron facets"  );

	gBallNumVertices[aLevel] = 6;
	for (i = 0; i < 6; i++)
		gBallVertices[aLevel][i] = v[i];

	gBallNumFacets[aLevel] = 8;
	for (i = 0; i < 8; i++)
		for (j = 0; j < 3; j++)
			gBallFacets[aLevel][i][j] = f[i][j];
}

static void SubdivideMesh(
	unsigned int	aSrcLevel)
{
	unsigned int	theDstLevel,
					theVertexCount,
					i,
					j,
					k;
	unsigned short	(*theNewVertexTable)[MAX_REFINED_BALL_VERTICES],
					v0,
					v1;
	double			theLengthSquared,
					theFactor;
	unsigned short	*v,
					vv[3];
	
	
	theDstLevel = aSrcLevel + 1;
	GEOMETRY_GAMES_ASSERT(theDstLevel <= MAX_BALL_REFINEMENT_LEVEL, "invalid source level gave invalid destination level");
	
	//	Subdivide the mesh, replacing each old facet with four new ones.
	//
	//		  /\		//
	//		 /__\		//
	//		/_\/_\		//
	//
	//	Be sure to keep the new facets together on the list,
	//	so the first 1/8 of the subdivision's facets give
	//	the 1/8 sphere, the first 1/4 give the 1/4 sphere,
	//	and the first 1/2 give the 1/2 sphere.

	//	Compute the number of vertices and facets in the destination mesh,
	//	using the reasoning explained in the comments that accompanying
	//	the definitions of MAX_REFINED_BALL_FACETS and MAX_REFINED_BALL_VERTICES
	//	in TorusGames-Common.h.
	gBallNumFacets[theDstLevel]		= (8 << (2 * theDstLevel));
	gBallNumVertices[theDstLevel]	= ((3 * gBallNumFacets[theDstLevel] + 12) / 6);
	GEOMETRY_GAMES_ASSERT(gBallNumFacets[theDstLevel]   <= MAX_REFINED_BALL_FACETS,   "too many ball facets"  );
	GEOMETRY_GAMES_ASSERT(gBallNumVertices[theDstLevel] <= MAX_REFINED_BALL_VERTICES, "too many ball vertices");
	
	
	//	First copy the source level's vertices to the destination level...
	for (i = 0; i < gBallNumVertices[aSrcLevel]; i++)
		gBallVertices[theDstLevel][i] = gBallVertices[aSrcLevel][i];
	theVertexCount = gBallNumVertices[aSrcLevel];

	//	...and then create one new vertex on each edge.
	//
	//	Use theNewVertexTable[][] to index them,
	//	so two facets sharing an edge can share the same vertex.
	//
	//	theNewVertexTable[v0][v1] takes the indices v0 and v1 of two old vertices,
	//	and gives the index of the new vertex that sits at the midpoint of the edge
	//	that connects v0 and v1.
	//
	//	The size of theNewVertexTable grows as the square of the number of source vertices.
	//	For modest meshes this won't be a problem.  For larger meshes a fancier algorithm,
	//	with linear rather than quadratic memory demands, could be used.
	//
	theNewVertexTable = (unsigned short	(*)[MAX_REFINED_BALL_VERTICES])
							GET_MEMORY(MAX_REFINED_BALL_VERTICES * MAX_REFINED_BALL_VERTICES * sizeof(unsigned short));
	GEOMETRY_GAMES_ASSERT(theNewVertexTable != NULL, "failed to allocate memory for theNewVertexTable");

	//	Initialize theNewVertexTable[][] to all 0xFFFF,
	//	to indicate that no new vertices have yet been created.
	for (i = 0; i < gBallNumVertices[aSrcLevel]; i++)
		for (j = 0; j < gBallNumVertices[aSrcLevel]; j++)
			theNewVertexTable[i][j] = 0xFFFF;

	//	For each edge in the source mesh, create a new vertex at its midpoint.
	for (i = 0; i < gBallNumFacets[aSrcLevel]; i++)
	{
		for (j = 0; j < 3; j++)
		{
			v0 = gBallFacets[aSrcLevel][i][   j   ];
			v1 = gBallFacets[aSrcLevel][i][(j+1)%3];

			if (theNewVertexTable[v0][v1] == 0xFFFF)
			{
				GEOMETRY_GAMES_ASSERT(	theVertexCount < gBallNumVertices[theDstLevel],
										"Internal error #1 in SubdivideMesh()");
				
				//	The new vertex will be midway between vertices v0 and v1.
				for (k = 0; k < 3; k++)
					gBallVertices[theDstLevel][theVertexCount].pos[k]
						= 0.5 * (gBallVertices[theDstLevel][v0].pos[k] + gBallVertices[theDstLevel][v1].pos[k]);

				//	Normalize the new vertex to unit length.
				theLengthSquared = 0.0;
				for (k = 0; k < 3; k++)
				{
					theLengthSquared += gBallVertices[theDstLevel][theVertexCount].pos[k]
									  * gBallVertices[theDstLevel][theVertexCount].pos[k];
				}
				GEOMETRY_GAMES_ASSERT(theLengthSquared >= 0.5, "Impossibly short interpolated vector");
				theFactor = 1.0 / sqrt(theLengthSquared);
				for (k = 0; k < 3; k++)
					gBallVertices[theDstLevel][theVertexCount].pos[k] *= theFactor;

				//	The new vertex's position may also serve as its normal vector.
				for (k = 0; k < 3; k++)
					gBallVertices[theDstLevel][theVertexCount].nor[k]
						= gBallVertices[theDstLevel][theVertexCount].pos[k];

				//	Record the new vertex at [v1][v0] as well as [v0][v1],
				//	so only one new vertex will get created for each edge.
				theNewVertexTable[v0][v1] = theVertexCount;
				theNewVertexTable[v1][v0] = theVertexCount;

				theVertexCount++;
			}
		}
	}
	GEOMETRY_GAMES_ASSERT(	theVertexCount == gBallNumVertices[theDstLevel],
							"Internal error #2 in SubdivideMesh()");

	//	Create the new faces.
	for (i = 0; i < gBallNumFacets[aSrcLevel]; i++)
	{
		//	The old vertices incident to this face will be v[0], v[1] and v[2].
		v = gBallFacets[aSrcLevel][i];

		//	The new vertices -- which sit at the midpoints of the old edges --
		//	will be vv[0], vv[1], and vv[2].
		//	Each vv[j] sits opposite the corresponding v[j].
		for (j = 0; j < 3; j++)
			vv[j] = theNewVertexTable[ v[(j+1)%3] ][ v[(j+2)%3] ];

		//	Create the new faces.

		gBallFacets[theDstLevel][4*i + 0][0] = vv[0];
		gBallFacets[theDstLevel][4*i + 0][1] = vv[1];
		gBallFacets[theDstLevel][4*i + 0][2] = vv[2];

		gBallFacets[theDstLevel][4*i + 1][0] = v[0];
		gBallFacets[theDstLevel][4*i + 1][1] = vv[2];
		gBallFacets[theDstLevel][4*i + 1][2] = vv[1];

		gBallFacets[theDstLevel][4*i + 2][0] = v[1];
		gBallFacets[theDstLevel][4*i + 2][1] = vv[0];
		gBallFacets[theDstLevel][4*i + 2][2] = vv[2];

		gBallFacets[theDstLevel][4*i + 3][0] = v[2];
		gBallFacets[theDstLevel][4*i + 3][1] = vv[1];
		gBallFacets[theDstLevel][4*i + 3][2] = vv[0];
	}
	
	FREE_MEMORY_SAFELY(theNewVertexTable);
}


void InitTubeMeshes(void)
{
	unsigned int	theRefinementLevel,
					n,
					i;
	double			theAngle;

	static bool		theTubeMeshesHaveBeenInitialized	= false;
	
	//	Design note:  We could have set the tube mesh up
	//	as a triangle strip rather than as a generic mesh.
	//	But given the cube and the balls are all generic meshes,
	//	I'm doing the tube the same way, to keep the code
	//	simple and consistent.  The Tic-Tac-Toe win line
	//	won't be a performance bottleneck in any case.
	
	//	The meshes need be initialized only once.
	if (theTubeMeshesHaveBeenInitialized)
		return;

	//	Unlike with the ball meshes, we may construct each tube mesh directly,
	//	without using any of the already constructed less refined meshes.
	for (theRefinementLevel = 0; theRefinementLevel <= MAX_TUBE_REFINEMENT_LEVEL; theRefinementLevel++)
	{
		//	Construct the tube in an arbitrary but well-defined position,
		//	namely running from -1 to +1 along the x axis,
		//	with cross-sectional radius 1 in the yz plane.

		//	Winding order is clockwise when viewed
		//	from outside the tube in a left-handed coordinate system.
		
		//	For brevity, define n to be the number of vertices,
		//	which also equals the number of faces.
		n = (8 << theRefinementLevel);

		gTubeNumVertices[theRefinementLevel] = n;
		for (i = 0; i < n; i++)
		{
			//	For many purposes it looks good to stagger the vertices, like this
			//
			//		theAngle = 2.0 * PI * (double)i / (double)n;
			//
			//	but to get tubes to fit precisely with their endcaps,
			//	it's easier to use non-staggered vertices.
			//
			theAngle = 2.0 * 2.0 * PI * (double)(i/2) / (double)n;

			gTubeVertices[theRefinementLevel][i].pos[0] = (i & 1) ? -1.0 : +1.0;
			gTubeVertices[theRefinementLevel][i].pos[1] = cos(theAngle);
			gTubeVertices[theRefinementLevel][i].pos[2] = sin(theAngle);

			gTubeVertices[theRefinementLevel][i].nor[0] = 0.0;
			gTubeVertices[theRefinementLevel][i].nor[1] = cos(theAngle);
			gTubeVertices[theRefinementLevel][i].nor[2] = sin(theAngle);
		}

		
		//	Create the faces in pairs.
		GEOMETRY_GAMES_ASSERT((n & 1) == 0, "number of facets must be even");
		gTubeNumFacets[theRefinementLevel] = n;
		for (i = 0; i < n; i += 2)	//	increment by +2 each time
		{
			gTubeFacets[theRefinementLevel][  i  ][0] = (i + 0) % n;
			gTubeFacets[theRefinementLevel][  i  ][1] = (i + 1) % n;
			gTubeFacets[theRefinementLevel][  i  ][2] = (i + 2) % n;

			gTubeFacets[theRefinementLevel][ i+1 ][0] = (i + 2) % n;
			gTubeFacets[theRefinementLevel][ i+1 ][1] = (i + 1) % n;
			gTubeFacets[theRefinementLevel][ i+1 ][2] = (i + 3) % n;
		}
	}
	
	//	The one-time mesh initialization is now complete.
	//	No need to repeat it if this function gets called again.
	theTubeMeshesHaveBeenInitialized = true;
}


void InitCircularSliceTriangleStrips(void)
{
	unsigned int						theRefinementLevel,
										n;
	TorusGames3DPolyhedronVertexData	*v;
	double								theCircumradius;
	unsigned int						theNumArcsPerSemicircle,	//	number of arcs into which each semi-circle gets divided
										i;
	double								theArcLength,
										theAngle,					//	angle measured from (+1,0,0)
										theCosine,
										theSine;

	static bool	theCircularSliceTriangleStripsHaveBeenInitialized	= false;
	
	//	The circular slice triangle strips need be initialized only once.
	if (theCircularSliceTriangleStripsHaveBeenInitialized)
		return;

	for (theRefinementLevel = 0; theRefinementLevel <= MAX_SLICE_REFINEMENT_LEVEL; theRefinementLevel++)
	{
		//	The circle lies in the z = 0 plane, with center at the origin and radius 1,
		//	and with its normal vector pointing in negative z direction.

		//	Winding order is clockwise when viewed
		//	from the "outward pointing" side in a left-handed coordinate system.

		//	Set the number of vertices for this refinement level
		gCircularSliceNumVertices[theRefinementLevel] = (4 << theRefinementLevel);
		
		//	For brevity...
		n = gCircularSliceNumVertices[theRefinementLevel];
		v = gCircularSliceVertices[theRefinementLevel];
		
		GEOMETRY_GAMES_ASSERT(n <= MAX_NUM_CIRCULAR_SLICE_VERTICES, "internal error: too many slice vertices");
		
		//	Let's approximate the circular slice not with an inscribed n-gon,
		//	but rather with a circumscribed n-gon, to ensure that it covers
		//	everything that it needs to cover, with no possibility of gaps.
		//	(By contrast, an inscribed n-gon could leave visible gaps
		//	in the space between the n-gon and the circle that it's approximating.)
		//
		//	For an n-gon with inradius 1.0, simple trigonometry
		//	suffices to compute its circumradius.
		theCircumradius = 1.0 / cos(PI / n);
		
		//	The above method for choosing n ensures that
		//	it is always even, and is always at least 4.
		//	The following construction relies on those facts.
		GEOMETRY_GAMES_ASSERT((n & 1) == 0, "n must be even");
		GEOMETRY_GAMES_ASSERT(n >= 4, "n must be at least 4");
		
		//	Place vertex  0  at (-R, 0, 0)
		v[ 0 ].pos[0] = (float) ( - theCircumradius );
		v[ 0 ].pos[1] =  0.0f;
		v[ 0 ].pos[2] =  0.0f;
		
		//	Place vertex n-1 at (+R, 0, 0)
		v[n-1].pos[0] = (float) ( + theCircumradius );
		v[n-1].pos[1] =  0.0f;
		v[n-1].pos[2] =  0.0f;
		
		//	Place vertices 1 through n-2 on the upper semi-circle (positive y)
		//	and lower semi-circle (negative y), in alternation,
		//	so that the resulting triangle strip forms a regular n-gon.
		//
		theNumArcsPerSemicircle	= n / 2;
		theArcLength			= PI / (double)theNumArcsPerSemicircle;
		for (i = 1; i <= theNumArcsPerSemicircle - 1; i++)
		{
			theAngle	= (theNumArcsPerSemicircle - i) * theArcLength;
			theCosine	= cos(theAngle);
			theSine		= sin(theAngle);
			
			v[2*i - 1].pos[0] = (float) (   theCosine * theCircumradius );
			v[2*i - 1].pos[1] = (float) ( + theSine   * theCircumradius );
			v[2*i - 1].pos[2] =  0.0f;
			
			v[  2*i  ].pos[0] = (float) (   theCosine * theCircumradius );
			v[  2*i  ].pos[1] = (float) ( - theSine   * theCircumradius );
			v[  2*i  ].pos[2] =  0.0f;
		}
		
		//	All the vertices have the same normal vector.
		for (i = 0; i < n; i++)
		{
			v[i].nor[0] =  0.0;
			v[i].nor[1] =  0.0;
			v[i].nor[2] = -1.0;
		}
	}
	
	//	The one-time slice initialization is now complete.
	//	No need to repeat it if this function gets called again.
	theCircularSliceTriangleStripsHaveBeenInitialized = true;
}


#ifdef __APPLE__
#pragma mark -
#pragma mark 3D geometrical utilities
#endif

void RotateXYZ(
	double	m[4][4])
{
	//	Postmultiply m by the matrix
	//
	//		0 1 0 0
	//		0 0 1 0
	//		1 0 0 0
	//		0 0 0 1
	//
	//	which rotates x → y → z → x.
	
	unsigned int	i;
	double			theSwap;
	
	for (i = 0; i < 4; i++)
	{
		theSwap = m[i][0];
		m[i][0] = m[i][2];
		m[i][2] = m[i][1];
		m[i][1] = theSwap;
	}
}

void RotateZYX(
	double	m[4][4])
{
	//	Postmultiply m by the matrix
	//
	//		0 0 1 0
	//		1 0 0 0
	//		0 1 0 0
	//		0 0 0 1
	//
	//	which rotates z → y → x → z.
	
	unsigned int	i;
	double			theSwap;
	
	for (i = 0; i < 4; i++)
	{
		theSwap = m[i][0];
		m[i][0] = m[i][1];
		m[i][1] = m[i][2];
		m[i][2] = theSwap;
	}
}

void HalfTurnY(
	double	m[4][4])
{
	//	Postmultiply m by the matrix
	//
	//		-1  0  0  0
	//		 0  1  0  0
	//		 0  0 -1  0
	//		 0  0  0  1
	//
	//	which rotates a half turn about the y axis.
	
	unsigned int	i;
	
	for (i = 0; i < 4; i++)
	{
		m[i][0] = - m[i][0];
		m[i][2] = - m[i][2];
	}
}

void SwapXY(
	double	m[4][4])
{
	//	Postmultiply m by the matrix
	//
	//		 0  1  0  0
	//		 1  0  0  0
	//		 0  0  1  0
	//		 0  0  0  1
	//
	//	which swaps the x and y axes.
	//	This reflection is useful because
	//
	//		- it takes the default 1/8 sphere x ≥ 0, y ≥ 0, z ≥ 0 to itself
	//		- it takes the default 1/4 sphere x ≥ 0, y ≥ 0 to itself
	//
	//	The caller applies this reflection to make the total number
	//	of reflections even, without changing the 1/8 sphere's 
	//	or 1/4 sphere's location.
	
	unsigned int	i;
	double			theSwap;
	
	for (i = 0; i < 4; i++)
	{
		theSwap = m[i][0];
		m[i][0] = m[i][1];
		m[i][1] = theSwap;
	}
}

void ReflectAxis(
	double			m[4][4],
	unsigned int	anAxis)	//	pass 0, 1 or 2 to reflect x, y or z axis
{
	unsigned int	i;
	
	for (i = 0; i < 4; i++)
		m[i][anAxis] = - m[i][anAxis];
}
